/*******************************************************************************

  
  Copyright(c) 2003 - 2005 Intel Corporation. All rights reserved.
  
  This program is free software; you can redistribute it and/or modify it 
  under the terms of the GNU General Public License as published by the Free 
  Software Foundation; either version 2 of the License, or (at your option) 
  any later version.
  
  This program is distributed in the hope that it will be useful, but WITHOUT 
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
  more details.
  
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 
  Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
  The full GNU General Public License is included in this distribution in the
  file called LICENSE.
  
  Contact Information:
  Linux NICS <linux.nics@intel.com>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497
  
  file: iamt_wd.c
  part of "Intel(R) Active Management Technology - KCS" Linux driver 

*******************************************************************************/

#include "iamt.h"

/**
 * setup_wd - sets up the structures for the WD commands
 * @device: the kcs device structure
 * 
 * Returns 0 on success, non-zero on failure.
 * NOTE: This function is called only in device initialization.
 */
 
int setup_wd(struct iamt_kcs_device *device)
{
    int err = 0;
    struct wd_start_request *request;
    device->wd_file = alloc_priv(NULL);

    if (!device->wd_file) {
        return -ENOMEM;
    }
    
    err = alloc_message(&device->wd_file->req,sizeof(struct wd_start_request));
    if (err) {
        kfree(device->wd_file);
        device->wd_file = NULL;
        return err;
    }
    err = alloc_message(&device->wd_file->res,sizeof(struct wd_start_response));
    if (err) {
        free_msg(&device->wd_file->req);
        kfree(device->wd_file);
        device->wd_file = NULL;
        return err;
    }

    request = (struct wd_start_request *)device->wd_file->req.msg;
    request->header.version.major = HOST_IF_MAJOR;
    request->header.version.minor = HOST_IF_MINOR;
    request->header.reserved = 0;
    request->header.command.cmd.value = START_WATCHDOG_REQUEST_CMD;
    request->header.length = sizeof(struct wd_start_request) - sizeof(struct pthi_header);
    request->timeout = wd_timeout;
    return 0;
}

/**
 * send_wd - send a WD command (prepare it to be sent and send if no other message currently being sent).
 * @data: A pointer to the device structure
 * 
 * NOTE: This function is called also by a timer
 */
 
void send_wd(unsigned long data)
{
    struct iamt_kcs_device *device = (struct iamt_kcs_device *)data;
    struct iamt_kcs_file_private *priv = device->wd_file;
    unsigned long flags;

    if (!priv) {
        return;
    }

    spin_lock_bh_irqsave(&device_lock, flags);
    spin_lock(&priv->file_lock);
    if (!device->active) {
        spin_unlock(&priv->file_lock);
        spin_unlock_bh_irqrestore(&device_lock, flags);
        return;
    }

    priv->res.size = sizeof(struct wd_start_response);
    priv->state = MESSAGE_RECEIVED;

    if (list_empty(&device->message_list)) {
        list_add(&priv->link, &device->message_list);
        device->curr_file = priv;
        iamt_start_transaction(device);
    }
    else {
        device->wd_pending = 1;
    }
    spin_unlock(&priv->file_lock);
    spin_unlock_bh_irqrestore(&device_lock, flags);
}

/**
 * stop_wd - send stop WD command to KCS
 * @device: the private device structure
 * 
 * NOTE: this function is called on device suht down.
 * The device->active flag will be zero and no locks will be taken.
 */
void stop_wd(struct iamt_kcs_device *device)
{
    struct wd_stop_request *request;
    unsigned long flags;


    device->curr_file = device->wd_file;
    spin_lock_bh_irqsave(&device_lock, flags);
    spin_lock(&device->curr_file->file_lock);
    device->wd_pending = 0;
    request = (struct wd_stop_request *)device->wd_file->req.msg;
           
    request->header.version.major = HOST_IF_MAJOR;
    request->header.version.minor = HOST_IF_MINOR;
    request->header.reserved = 0;
    request->header.command.cmd.value = STOP_WATCHDOG_REQUEST_CMD;
    request->header.length = sizeof(struct wd_stop_request) - sizeof(struct pthi_header);

    list_add(&device->wd_file->link, &device->message_list);
    
    iamt_start_transaction(device);
    spin_unlock(&device->curr_file->file_lock);
    spin_unlock_bh_irqrestore(&device_lock, flags);

    /* wait for stop WD command to finish */
    wait_event_interruptible(device->wd_file->wait, device->wd_file->state == MESSAGE_WAITING);
}

/**
 * free_wd - free the WD structures on device shut down.
 * @device: The device structure
 */

void free_wd(struct iamt_kcs_device *device)
{
    free_msg(&device->wd_file->req);
    free_msg(&device->wd_file->res);
    kfree(device->wd_file);
    device->wd_file = NULL;
}

